%
% Simulate solutions to budget neutral employment subsidy
% Date: 23 May 2018
%
% Purpose: Simulate solutions to the employment subsidy equations across
%          a range of potential elasticities

% Housekeeping

% NOTE: You must change the folder to the location of this script. Do not
% simply add it as a path.
cd(fileparts(which(mfilename)))
cd ../..

close all
clear all
clc
rng default  % For reproducibility

% Set up variables
Eps_Range = linspace(0.0,0.5,501);
Target_NonEmp = 0.20;
Gamma_Range = [0.5, 1, 2];
Externality = 0.363;
Benefit = 0.6;
Sol_x = NaN(3,501);

% Solve equation
for i = 1:3
    Gamma = Gamma_Range(1,i);
    for j = 1:501
        Eps = Eps_Range(1,j);
        syms x;
        eqn = Benefit - x - (1 + (Target_NonEmp/(1-Target_NonEmp))*x)*((1 + (Eps/Target_NonEmp)*(x/(1-Target_NonEmp) - Externality))^(1/Gamma)) == 0;
        Sol_x(i,j) = vpasolve(eqn,x,[-0.35,1]);
    end
end

% Save dataset
Eps = Eps_Range';
Gamma0_5 = Sol_x(1,:)';
Gamma1 = Sol_x(2,:)';
Gamma2 = Sol_x(3,:)';
T = table(Eps,Gamma0_5,Gamma1,Gamma2);
writetable(T,'output/figures/xlsx/figure20.xlsx','Sheet',1,'Range','A1')

% Construct plot
plot(Eps_Range,Sol_x)
legend('Gamma = 0.5','Gamma = 1.0','Gamma = 2.0','Location','northwest')
xlabel('Elasticity of employment rate')
ylabel('Size of optimal benefits twist relative to wage')

print('output/figures/gph/figure20','-dpng')
print('output/figures/eps/figure20','-depsc')